# Field Selector Guide

## How the Extension Finds Form Fields

The extension uses a **multi-strategy approach** to find form fields, making it resilient to form structure changes.

## Strategy 1: Multiple CSS Selectors

Each field has an array of CSS selectors that are tried in order:

```javascript
employerName: {
  selectors: [
    '#txt_employer_name',                    // Try exact ID first
    'input[name*="employer" i][name*="name" i]',  // Try name attributes
    'input[id*="employer" i][id*="name" i]',      // Try partial ID match
  ],
  // ...
}
```

**Why multiple selectors?**
- Form fields may have encrypted or dynamic IDs
- Field names might vary slightly
- Forms may be updated over time

## Strategy 2: Label-Based Search

If all CSS selectors fail, the extension searches by label text:

```javascript
findByLabel: 'employer name'
```

This function:
1. Searches for `<label>` elements containing the text
2. Finds associated input via `for` attribute or DOM structure
3. Searches for inputs with matching `placeholder` or `aria-label`
4. Looks for inputs near label text in the DOM tree

## Strategy 3: Fallback Patterns

For dynamic forms, the extension tries:
- Partial name matching (case-insensitive)
- Partial ID matching
- Type-based search (input, select, textarea)
- Sibling/parent relationship search

## Adding New Selectors

If a field isn't filling, follow these steps:

### Step 1: Inspect the Field

1. Open the form page
2. Right-click the field → "Inspect"
3. In DevTools, note:
   - `id` attribute: `id="actual_id_here"`
   - `name` attribute: `name="actual_name_here"`
   - `class` attribute: `class="actual_class_here"`
   - Nearby `<label>` text

### Step 2: Add to FIELD_MAP

Open `lib/field-mappings.js` and find the field:

```javascript
employerName: {
  selectors: [
    '#txt_employer_name',  // Existing selector
    // Add your new selector here:
    '#actual_id_here',     // Exact ID
    'input[name="actual_name_here"]',  // Exact name
    'input.actual_class_here',  // Class selector
  ],
  type: 'text',
  findByLabel: 'employer name'
}
```

### Step 3: Test

1. Reload the extension in `chrome://extensions`
2. Refresh the form page
3. Click "Auto-Fill"
4. Check Console for success/failure messages

## Common Selector Patterns

### Text Inputs
```javascript
'#field_id'                    // Exact ID
'input[name="field_name"]'     // Exact name
'input[name*="field" i]'       // Partial name (case-insensitive)
'input[type="text"]'           // Type-based
```

### Select Dropdowns
```javascript
'select#field_id'
'select[name="field_name"]'
'select[name*="field" i]'
```

### Checkboxes/Radio Buttons
```javascript
'input[type="checkbox"]#field_id'
'input[type="radio"][name="field_name"]'
'input[type="checkbox"][name*="field" i]'
```

### Textareas
```javascript
'textarea#field_id'
'textarea[name="field_name"]'
```

## Debugging Tips

### Enable Console Logging

The extension logs all field-finding attempts. Check the Console:

```
Silent Whistle: Starting auto-fill with data: {...}
Silent Whistle: Filled 15 fields
Silent Whistle: Failed to fill: ['Employer Name', 'Job Title']
```

### Test Individual Selectors

In the Console, test selectors manually:

```javascript
// Test if selector works
document.querySelector('#txt_employer_name')

// Test label-based search
window.findFieldByLabel('employer name', 'input')
```

### Check Field Visibility

Some fields might be hidden or in iframes:

```javascript
// Check if field is visible
const field = document.querySelector('#field_id');
console.log('Visible:', field.offsetParent !== null);
console.log('Display:', window.getComputedStyle(field).display);
```

## Form Structure Variations

The MA AGO form may have different structures:

1. **Standard HTML Form**: Direct field access
2. **JavaScript-Rendered**: Fields created dynamically
3. **Iframe-Based**: Fields in nested iframes
4. **Encrypted IDs**: Field IDs change on each load

The multi-strategy approach handles most of these cases automatically.

## Best Practices

1. **Add multiple selectors** for each field (don't rely on one)
2. **Use partial matching** (`name*="field"`) for dynamic names
3. **Include label-based search** as fallback
4. **Test after form updates** - forms may change
5. **Log failures** to identify patterns

## Example: Complete Field Mapping

```javascript
employerName: {
  selectors: [
    // Strategy 1: Exact matches (most reliable)
    '#txt_employer_name',
    'input[name="employerName"]',
    
    // Strategy 2: Partial matches (for dynamic names)
    'input[name*="employer" i][name*="name" i]',
    'input[id*="employer" i][id*="name" i]',
    
    // Strategy 3: Type + attribute combination
    'input[type="text"][name*="employer" i]',
  ],
  type: 'text',
  findByLabel: 'employer name'  // Fallback strategy
}
```

This ensures the field is found even if:
- The exact ID changes
- The name attribute is slightly different
- The form structure is updated
- The field is in a different location


