// content.js - Injected into the MA AGO form page

(function() {
  'use strict';
  
  // Prevent double-injection
  if (window.silentWhistleInjected) return;
  window.silentWhistleInjected = true;
  
  console.log('Silent Whistle: Extension loaded on form page');
  
  // ==========================================
  // CONFIGURATION
  // ==========================================
  
  const SIDEBAR_WIDTH = 380;
  
  // ==========================================
  // LOAD USER DATA FROM STORAGE
  // ==========================================
  
  let userData = null;
  
  async function loadUserData() {
    return new Promise((resolve) => {
      chrome.storage.local.get(['silentWhistleData'], (result) => {
        userData = result.silentWhistleData || getDefaultData();
        resolve(userData);
      });
    });
  }
  
  function getDefaultData() {
    // Default/demo data - in production, this comes from your main app
    return {
      employerName: '',
      employerType: 'Corporation',
      employerStreet: '',
      employerCity: '',
      employerState: 'Massachusetts',
      employerZip: '',
      employerPhone: '',
      numberOfEmployees: '',
      employmentStartDate: '',
      employmentEndDate: '',
      currentlyEmployed: false,
      violationStartDate: '',
      violationEndDate: '',
      violationOngoing: false,
      violations: [],
      requestPRA: false,
      jobTitle: '',
      payRate: '',
      payPeriod: 'Hour',
      totalAmountOwed: '',
      workInMA: true,
      detailedDescription: '',
      firstName: '',
      lastName: '',
      street: '',
      city: '',
      state: 'Massachusetts',
      zip: '',
      phone: '',
      email: ''
    };
  }
  
  // ==========================================
  // AUTO-FILL FUNCTIONS (Enhanced with multiple selector strategies)
  // ==========================================
  
  /**
   * Find field using multiple selector strategies
   */
  function findField(fieldConfig) {
    // Strategy 1: Try all selectors in order
    if (fieldConfig.selectors && Array.isArray(fieldConfig.selectors)) {
      for (const selector of fieldConfig.selectors) {
        try {
          const field = document.querySelector(selector);
          if (field) return field;
        } catch (e) {
          continue;
        }
      }
    }
    
    // Strategy 2: Try single selector (backward compatibility)
    if (fieldConfig.selector) {
      try {
        const field = document.querySelector(fieldConfig.selector);
        if (field) return field;
      } catch (e) {
        // Continue to next strategy
      }
    }
    
    // Strategy 3: Find by label text (if available)
    if (fieldConfig.findByLabel && typeof window.findFieldByLabel === 'function') {
      try {
        const fieldType = fieldConfig.type === 'select' ? 'select' : 
                         fieldConfig.type === 'textarea' ? 'textarea' : 'input';
        const field = window.findFieldByLabel(fieldConfig.findByLabel, fieldType);
        if (field) return field;
      } catch (e) {
        console.warn('findFieldByLabel failed:', e);
      }
    }
    
    return null;
  }
  
  function fillTextField(fieldConfig, value) {
    const field = findField(fieldConfig);
    if (field && value) {
      field.value = value;
      // Trigger change event so the form recognizes the input
      field.dispatchEvent(new Event('input', { bubbles: true }));
      field.dispatchEvent(new Event('change', { bubbles: true }));
      highlightField(field, 'success');
      return true;
    }
    return false;
  }
  
  function fillSelectField(fieldConfig, value) {
    const field = findField(fieldConfig);
    if (field && value) {
      // Find option by value or text
      const options = Array.from(field.options);
      const option = options.find(opt => 
        opt.value.toLowerCase() === value.toLowerCase() ||
        opt.text.toLowerCase().includes(value.toLowerCase())
      );
      if (option) {
        field.value = option.value;
        field.dispatchEvent(new Event('change', { bubbles: true }));
        highlightField(field, 'success');
        return true;
      }
    }
    return false;
  }
  
  function fillCheckbox(fieldConfig, shouldCheck) {
    const field = findField(fieldConfig);
    if (field) {
      if (field.checked !== shouldCheck) {
        field.click();
      }
      highlightField(field.parentElement || field, 'success');
      return true;
    }
    return false;
  }
  
  function clickRadio(fieldConfig) {
    const field = findField(fieldConfig);
    if (field) {
      field.click();
      highlightField(field.parentElement || field, 'success');
      return true;
    }
    return false;
  }
  
  function highlightField(element, type) {
    if (!element) return;
    const color = type === 'success' ? '#22c55e' : '#ef4444';
    const originalBorder = element.style.border;
    element.style.border = `3px solid ${color}`;
    element.style.transition = 'border 0.3s ease';
    setTimeout(() => {
      element.style.border = originalBorder;
    }, 2000);
  }
  
  // ==========================================
  // MAIN AUTO-FILL FUNCTION
  // ==========================================
  
  async function autoFillForm() {
    if (!userData) {
      await loadUserData();
    }
    
    console.log('Silent Whistle: Starting auto-fill with data:', userData);
    
    let filledCount = 0;
    let failedFields = [];
    
    // Helper to try filling and track results
    function tryFill(fieldName, fillFn) {
      try {
        if (fillFn()) {
          filledCount++;
        } else {
          failedFields.push(fieldName);
        }
      } catch (e) {
        console.error(`Failed to fill ${fieldName}:`, e);
        failedFields.push(fieldName);
      }
    }
    
    // === SECTION: Request Type ===
    tryFill('Request Type', () => clickRadio(FIELD_MAP.requestType));
    
    // === SECTION: Violations ===
    if (userData.violations && userData.violations.length > 0) {
      userData.violations.forEach(violation => {
        const key = violation.toLowerCase().replace(/[^a-z]/g, '');
        if (FIELD_MAP.violations[key]) {
          tryFill(`Violation: ${violation}`, () => 
            fillCheckbox(FIELD_MAP.violations[key], true)
          );
        }
      });
    }
    
    // === SECTION: Private Right of Action ===
    if (userData.requestPRA) {
      tryFill('PRA', () => fillCheckbox(FIELD_MAP.privateRightOfAction, true));
    }
    
    // === SECTION: Employer Information ===
    tryFill('Employer Name', () => fillTextField(FIELD_MAP.employerName, userData.employerName));
    tryFill('Employer Type', () => fillSelectField(FIELD_MAP.employerType, userData.employerType));
    tryFill('Employer Street', () => fillTextField(FIELD_MAP.employerStreet, userData.employerStreet));
    tryFill('Employer City', () => fillTextField(FIELD_MAP.employerCity, userData.employerCity));
    tryFill('Employer State', () => fillSelectField(FIELD_MAP.employerState, userData.employerState));
    tryFill('Employer Zip', () => fillTextField(FIELD_MAP.employerZip, userData.employerZip));
    tryFill('Employer Phone', () => fillTextField(FIELD_MAP.employerPhone, userData.employerPhone));
    tryFill('Number of Employees', () => fillTextField(FIELD_MAP.numberOfEmployees, userData.numberOfEmployees));
    
    // === SECTION: Employment Dates ===
    tryFill('Employment Start', () => fillTextField(FIELD_MAP.employmentStartDate, userData.employmentStartDate));
    if (!userData.currentlyEmployed) {
      tryFill('Employment End', () => fillTextField(FIELD_MAP.employmentEndDate, userData.employmentEndDate));
    } else {
      tryFill('Currently Employed', () => fillCheckbox(FIELD_MAP.currentlyEmployed, true));
    }
    
    // === SECTION: Violation Period ===
    tryFill('Violation Start', () => fillTextField(FIELD_MAP.violationStartDate, userData.violationStartDate));
    if (!userData.violationOngoing) {
      tryFill('Violation End', () => fillTextField(FIELD_MAP.violationEndDate, userData.violationEndDate));
    } else {
      tryFill('Violation Ongoing', () => fillCheckbox(FIELD_MAP.violationOngoing, true));
    }
    
    // === SECTION: Job Details ===
    tryFill('Job Title', () => fillTextField(FIELD_MAP.jobTitle, userData.jobTitle));
    tryFill('Pay Rate', () => fillTextField(FIELD_MAP.payRate, userData.payRate));
    tryFill('Pay Period', () => fillSelectField(FIELD_MAP.payPeriod, userData.payPeriod));
    tryFill('Amount Owed', () => fillTextField(FIELD_MAP.totalAmountOwed, userData.totalAmountOwed));
    tryFill('Work in MA', () => fillSelectField(FIELD_MAP.workInMA, userData.workInMA ? 'Yes' : 'No'));
    tryFill('Description', () => fillTextField(FIELD_MAP.detailedDescription, userData.detailedDescription));
    
    // === SECTION: Contact Information ===
    tryFill('First Name', () => fillTextField(FIELD_MAP.firstName, userData.firstName));
    tryFill('Last Name', () => fillTextField(FIELD_MAP.lastName, userData.lastName));
    tryFill('Street', () => fillTextField(FIELD_MAP.street, userData.street));
    tryFill('City', () => fillTextField(FIELD_MAP.city, userData.city));
    tryFill('State', () => fillSelectField(FIELD_MAP.state, userData.state));
    tryFill('Zip', () => fillTextField(FIELD_MAP.zip, userData.zip));
    tryFill('Phone', () => fillTextField(FIELD_MAP.phone, userData.phone));
    tryFill('Email', () => fillTextField(FIELD_MAP.email, userData.email));
    
    // === IMPORTANT: DO NOT AUTO-FILL SIGNATURE ===
    // The user MUST fill the signature section themselves
    
    // Report results
    console.log(`Silent Whistle: Filled ${filledCount} fields`);
    if (failedFields.length > 0) {
      console.warn('Silent Whistle: Failed to fill:', failedFields);
    }
    
    return { filledCount, failedFields };
  }
  
  // ==========================================
  // CREATE SIDEBAR UI
  // ==========================================
  
  function createSidebar() {
    // Shrink the page to make room for sidebar
    document.body.style.marginRight = `${SIDEBAR_WIDTH}px`;
    document.body.style.transition = 'margin-right 0.3s ease';
    
    // Create sidebar container
    const sidebar = document.createElement('div');
    sidebar.id = 'sw-sidebar';
    sidebar.innerHTML = `
      <div class="sw-header">
        <div class="sw-header-content">
          <div class="sw-logo">📋 Silent Whistle</div>
          <div class="sw-header-actions">
            <button class="sw-btn-icon" id="sw-minimize">−</button>
            <button class="sw-btn-icon" id="sw-close">×</button>
          </div>
        </div>
        <p class="sw-subtitle">Wage Complaint Assistant</p>
      </div>
      
      <div class="sw-body">
        <div class="sw-section">
          <h3>🚀 Quick Actions</h3>
          <button class="sw-btn sw-btn-primary" id="sw-autofill">
            ⚡ Auto-Fill All Fields
          </button>
          <p class="sw-hint">Fills the form with your saved information</p>
        </div>
        
        <div class="sw-section">
          <h3>⚠️ You Must Complete</h3>
          <div class="sw-warning-box">
            <p>After auto-fill, you MUST:</p>
            <ol>
              <li>✓ Review all filled information</li>
              <li>✓ Check the certification box</li>
              <li>✓ Type your full legal name</li>
              <li>✓ Click the Submit button</li>
            </ol>
          </div>
        </div>
        
        <div class="sw-section">
          <h3>📊 Fill Status</h3>
          <div id="sw-status" class="sw-status-box">
            Click "Auto-Fill" to begin
          </div>
        </div>
        
        <div class="sw-section">
          <h3>🛡️ Safety Notice</h3>
          <div class="sw-safety-box">
            The AG's Office does NOT share information with ICE. 
            All workers can file regardless of immigration status.
          </div>
        </div>
      </div>
    `;
    
    document.body.appendChild(sidebar);
    
    // Event Listeners
    document.getElementById('sw-autofill').addEventListener('click', async () => {
      const statusEl = document.getElementById('sw-status');
      statusEl.innerHTML = '⏳ Filling form...';
      statusEl.className = 'sw-status-box sw-status-loading';
      
      const result = await autoFillForm();
      
      if (result.failedFields.length === 0) {
        statusEl.innerHTML = `✅ Successfully filled ${result.filledCount} fields!<br><br>
          <strong>Now scroll down to:</strong><br>
          1. Check the certification box<br>
          2. Type your name<br>
          3. Click Submit`;
        statusEl.className = 'sw-status-box sw-status-success';
      } else {
        statusEl.innerHTML = `⚠️ Filled ${result.filledCount} fields.<br>
          Could not find: ${result.failedFields.slice(0, 3).join(', ')}${result.failedFields.length > 3 ? '...' : ''}<br>
          Please fill these manually.`;
        statusEl.className = 'sw-status-box sw-status-warning';
      }
      
      // Scroll to signature section
      const signatureSection = findField(FIELD_MAP.certificationCheckbox);
      if (signatureSection) {
        signatureSection.scrollIntoView({ behavior: 'smooth', block: 'center' });
      }
    });
    
    document.getElementById('sw-minimize').addEventListener('click', () => {
      const body = document.querySelector('.sw-body');
      body.style.display = body.style.display === 'none' ? 'block' : 'none';
    });
    
    document.getElementById('sw-close').addEventListener('click', () => {
      sidebar.remove();
      document.body.style.marginRight = '0';
    });
  }
  
  // ==========================================
  // INITIALIZE
  // ==========================================
  
  async function init() {
    await loadUserData();
    createSidebar();
    console.log('Silent Whistle: Sidebar created');
  }
  
  // Wait for page to be fully loaded
  if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', init);
  } else {
    init();
  }
  
})();

