// lib/field-mappings.js
// Form field selectors for MA Attorney General wage complaint form
// NOTE: This form uses dynamic/encrypted field names, so we use multiple selector strategies

// Helper function to find fields by label text or placeholder
function findFieldByLabel(labelText, fieldType = 'input') {
  // Try multiple strategies
  const strategies = [
    // Strategy 1: Find label with text, then get associated input
    () => {
      const labels = Array.from(document.querySelectorAll('label'));
      const label = labels.find(l => l.textContent.toLowerCase().includes(labelText.toLowerCase()));
      if (label) {
        const forAttr = label.getAttribute('for');
        if (forAttr) return document.getElementById(forAttr) || document.querySelector(`[name="${forAttr}"]`);
        const input = label.querySelector(fieldType);
        if (input) return input;
      }
    },
    // Strategy 2: Find input with placeholder containing text
    () => {
      const inputs = Array.from(document.querySelectorAll(fieldType));
      return inputs.find(inp => {
        const placeholder = inp.getAttribute('placeholder') || '';
        return placeholder.toLowerCase().includes(labelText.toLowerCase());
      });
    },
    // Strategy 3: Find input near label text (sibling or parent)
    () => {
      const allText = Array.from(document.querySelectorAll('*'));
      const element = allText.find(el => {
        const text = el.textContent || '';
        return text.toLowerCase().includes(labelText.toLowerCase()) && 
               (el.tagName === 'LABEL' || el.tagName === 'SPAN' || el.tagName === 'DIV');
      });
      if (element) {
        // Look for input in parent or siblings
        const parent = element.closest('div, tr, td, li');
        if (parent) {
          return parent.querySelector(fieldType) || 
                 parent.querySelector('input, select, textarea');
        }
      }
    },
    // Strategy 4: Find by aria-label
    () => {
      return document.querySelector(`${fieldType}[aria-label*="${labelText}" i]`);
    }
  ];
  
  for (const strategy of strategies) {
    try {
      const field = strategy();
      if (field) return field;
    } catch (e) {
      continue;
    }
  }
  return null;
}

const FIELD_MAP = {
  // === SECTION: About Your Request ===
  requestType: {
    // Multiple selector strategies for dynamic forms
    selectors: [
      'input[value*="Non-Payment" i]',
      'input[value*="Wage" i][type="radio"]',
      'input[type="radio"][name*="request" i]',
      'input[type="radio"][name*="complaint" i]'
    ],
    type: 'radio',
    action: 'click',
    findByLabel: 'non-payment of wage'
  },
  
  // === SECTION: Reasons for Filing ===
  violations: {
    // Multiple selector strategies for each violation type
    nonPayment: { 
      selectors: [
        '#chk_nonpayment',
        'input[type="checkbox"][name*="nonpayment" i]',
        'input[type="checkbox"][name*="unpaid" i]',
        'input[type="checkbox"][id*="nonpayment" i]'
      ],
      type: 'checkbox',
      findByLabel: 'non-payment'
    },
    overtime: { 
      selectors: [
        '#chk_overtime',
        'input[type="checkbox"][name*="overtime" i]',
        'input[type="checkbox"][id*="overtime" i]'
      ],
      type: 'checkbox',
      findByLabel: 'overtime'
    },
    tips: { 
      selectors: [
        '#chk_tips',
        'input[type="checkbox"][name*="tip" i]',
        'input[type="checkbox"][id*="tip" i]'
      ],
      type: 'checkbox',
      findByLabel: 'tip'
    },
    minimumWage: { 
      selectors: [
        '#chk_minwage',
        'input[type="checkbox"][name*="minimum" i]',
        'input[type="checkbox"][id*="minimum" i]'
      ],
      type: 'checkbox',
      findByLabel: 'minimum wage'
    },
    finalPay: { 
      selectors: [
        '#chk_finalpay',
        'input[type="checkbox"][name*="final" i]',
        'input[type="checkbox"][id*="final" i]'
      ],
      type: 'checkbox',
      findByLabel: 'final pay'
    },
    other: { 
      selectors: [
        '#chk_other',
        'input[type="checkbox"][name*="other" i]',
        'input[type="checkbox"][id*="other" i]'
      ],
      type: 'checkbox',
      findByLabel: 'other'
    }
  },
  
  privateRightOfAction: {
    selectors: [
      '#chk_pra',
      'input[type="checkbox"][name*="pra" i]',
      'input[type="checkbox"][name*="private" i]',
      'input[type="checkbox"][id*="pra" i]'
    ],
    type: 'checkbox',
    findByLabel: 'private right of action'
  },
  
  // === SECTION: Filing a Complaint Against (Employer Info) ===
  employerName: {
    selectors: [
      '#txt_employer_name',
      'input[name*="employer" i][name*="name" i]',
      'input[name*="employerName" i]',
      'input[id*="employer" i][id*="name" i]'
    ],
    type: 'text',
    findByLabel: 'employer name'
  },
  employerType: {
    selectors: [
      '#ddl_employer_type',
      'select[name*="employer" i][name*="type" i]',
      'select[id*="employer" i][id*="type" i]'
    ],
    type: 'select',
    findByLabel: 'employer type'
  },
  employerStreet: {
    selectors: [
      '#txt_employer_street',
      'input[name*="employer" i][name*="street" i]',
      'input[name*="employer" i][name*="address" i]',
      'input[id*="employer" i][id*="street" i]'
    ],
    type: 'text',
    findByLabel: 'employer street'
  },
  employerCity: {
    selectors: [
      '#txt_employer_city',
      'input[name*="employer" i][name*="city" i]',
      'input[id*="employer" i][id*="city" i]'
    ],
    type: 'text',
    findByLabel: 'employer city'
  },
  employerState: {
    selectors: [
      '#ddl_employer_state',
      'select[name*="employer" i][name*="state" i]',
      'select[id*="employer" i][id*="state" i]'
    ],
    type: 'select',
    findByLabel: 'employer state'
  },
  employerZip: {
    selectors: [
      '#txt_employer_zip',
      'input[name*="employer" i][name*="zip" i]',
      'input[name*="employer" i][name*="postal" i]',
      'input[id*="employer" i][id*="zip" i]'
    ],
    type: 'text',
    findByLabel: 'employer zip'
  },
  employerPhone: {
    selectors: [
      '#txt_employer_phone',
      'input[name*="employer" i][name*="phone" i]',
      'input[type="tel"][name*="employer" i]',
      'input[id*="employer" i][id*="phone" i]'
    ],
    type: 'text',
    findByLabel: 'employer phone'
  },
  numberOfEmployees: {
    selectors: [
      '#txt_num_employees',
      'input[name*="employee" i][name*="number" i]',
      'input[name*="number" i][name*="employee" i]',
      'input[id*="employee" i][id*="number" i]'
    ],
    type: 'text',
    findByLabel: 'number of employees'
  },
  
  // === SECTION: Employment Dates ===
  employmentStartDate: {
    selectors: [
      '#txt_emp_start_date',
      'input[name*="start" i][name*="date" i]',
      'input[name*="employment" i][name*="start" i]',
      'input[type="date"][name*="start" i]',
      'input[id*="start" i][id*="date" i]'
    ],
    type: 'date',
    findByLabel: 'employment start date'
  },
  employmentEndDate: {
    selectors: [
      '#txt_emp_end_date',
      'input[name*="end" i][name*="date" i]',
      'input[name*="employment" i][name*="end" i]',
      'input[type="date"][name*="end" i]',
      'input[id*="end" i][id*="date" i]'
    ],
    type: 'date',
    findByLabel: 'employment end date'
  },
  currentlyEmployed: {
    selectors: [
      '#chk_currently_employed',
      'input[type="checkbox"][name*="currently" i]',
      'input[type="checkbox"][name*="current" i][name*="employ" i]',
      'input[type="checkbox"][id*="currently" i]'
    ],
    type: 'checkbox',
    findByLabel: 'currently employed'
  },
  
  // === SECTION: Violation Period ===
  violationStartDate: {
    selectors: [
      '#txt_vio_start_date',
      'input[name*="violation" i][name*="start" i]',
      'input[name*="violation" i][name*="date" i]',
      'input[type="date"][name*="violation" i]',
      'input[id*="violation" i][id*="start" i]'
    ],
    type: 'date',
    findByLabel: 'violation start date'
  },
  violationEndDate: {
    selectors: [
      '#txt_vio_end_date',
      'input[name*="violation" i][name*="end" i]',
      'input[type="date"][name*="violation" i][name*="end" i]',
      'input[id*="violation" i][id*="end" i]'
    ],
    type: 'date',
    findByLabel: 'violation end date'
  },
  violationOngoing: {
    selectors: [
      '#chk_vio_ongoing',
      'input[type="checkbox"][name*="violation" i][name*="ongoing" i]',
      'input[type="checkbox"][id*="violation" i][id*="ongoing" i]'
    ],
    type: 'checkbox',
    findByLabel: 'violation ongoing'
  },
  
  // === SECTION: Complaint Details ===
  helpFor: {
    selectors: [
      '#ddl_help_for',
      'select[name*="help" i]',
      'select[name*="filing" i][name*="for" i]',
      'select[id*="help" i]'
    ],
    type: 'select',
    value: 'myself',
    findByLabel: 'filing for'
  },
  jobTitle: {
    selectors: [
      '#txt_job_title',
      'input[name*="job" i][name*="title" i]',
      'input[name*="position" i]',
      'input[id*="job" i][id*="title" i]'
    ],
    type: 'text',
    findByLabel: 'job title'
  },
  payRate: {
    selectors: [
      '#txt_pay_rate',
      'input[name*="pay" i][name*="rate" i]',
      'input[name*="wage" i]',
      'input[name*="hourly" i]',
      'input[id*="pay" i][id*="rate" i]'
    ],
    type: 'text',
    findByLabel: 'pay rate'
  },
  payPeriod: {
    selectors: [
      '#ddl_pay_period',
      'select[name*="pay" i][name*="period" i]',
      'select[name*="frequency" i]',
      'select[id*="pay" i][id*="period" i]'
    ],
    type: 'select',
    findByLabel: 'pay period'
  },
  totalAmountOwed: {
    selectors: [
      '#txt_amount_owed',
      'input[name*="amount" i][name*="owed" i]',
      'input[name*="owed" i]',
      'input[id*="amount" i][id*="owed" i]'
    ],
    type: 'text',
    findByLabel: 'amount owed'
  },
  workInMA: {
    selectors: [
      '#ddl_work_in_ma',
      'select[name*="work" i][name*="massachusetts" i]',
      'select[name*="work" i][name*="ma" i]',
      'select[id*="work" i][id*="ma" i]'
    ],
    type: 'select',
    findByLabel: 'work in massachusetts'
  },
  detailedDescription: {
    selectors: [
      '#txt_description',
      'textarea[name*="description" i]',
      'textarea[name*="details" i]',
      'textarea[id*="description" i]',
      'textarea[id*="details" i]'
    ],
    type: 'textarea',
    findByLabel: 'description'
  },
  
  // === SECTION: Employee Contact Information ===
  firstName: {
    selectors: [
      '#txt_first_name',
      'input[name*="first" i][name*="name" i]',
      'input[name*="firstName" i]',
      'input[id*="first" i][id*="name" i]'
    ],
    type: 'text',
    findByLabel: 'first name'
  },
  lastName: {
    selectors: [
      '#txt_last_name',
      'input[name*="last" i][name*="name" i]',
      'input[name*="lastName" i]',
      'input[id*="last" i][id*="name" i]'
    ],
    type: 'text',
    findByLabel: 'last name'
  },
  street: {
    selectors: [
      '#txt_street',
      'input[name*="street" i]',
      'input[name*="address" i][name*="line" i]',
      'input[id*="street" i]'
    ],
    type: 'text',
    findByLabel: 'street address'
  },
  city: {
    selectors: [
      '#txt_city',
      'input[name*="city" i]',
      'input[id*="city" i]'
    ],
    type: 'text',
    findByLabel: 'city'
  },
  state: {
    selectors: [
      '#ddl_state',
      'select[name*="state" i]',
      'select[id*="state" i]'
    ],
    type: 'select',
    findByLabel: 'state'
  },
  zip: {
    selectors: [
      '#txt_zip',
      'input[name*="zip" i]',
      'input[name*="postal" i]',
      'input[id*="zip" i]'
    ],
    type: 'text',
    findByLabel: 'zip code'
  },
  phone: {
    selectors: [
      '#txt_phone',
      'input[name*="phone" i]',
      'input[type="tel"]',
      'input[id*="phone" i]'
    ],
    type: 'text',
    findByLabel: 'phone'
  },
  email: {
    selectors: [
      '#txt_email',
      'input[name*="email" i]',
      'input[type="email"]',
      'input[id*="email" i]'
    ],
    type: 'text',
    findByLabel: 'email'
  },
  
  // === SECTION: Signature (USER MUST FILL THIS) ===
  // DO NOT AUTO-FILL THESE - legal requirement
  certificationCheckbox: {
    selectors: [
      '#chk_certification',
      'input[type="checkbox"][name*="certif" i]',
      'input[type="checkbox"][name*="agree" i]',
      'input[type="checkbox"][id*="certif" i]'
    ],
    type: 'checkbox',
    autoFill: false, // User must check this themselves
    findByLabel: 'certification'
  },
  signatureName: {
    selectors: [
      '#txt_signature',
      'input[name*="signature" i]',
      'input[name*="sign" i]',
      'input[id*="signature" i]'
    ],
    type: 'text',
    autoFill: false, // User must type their name themselves
    findByLabel: 'signature'
  }
};

// Export helper function
if (typeof window !== 'undefined') {
  window.FIELD_MAP = FIELD_MAP;
  window.findFieldByLabel = findFieldByLabel;
}

// Export for use in content.js
if (typeof window !== 'undefined') {
  window.FIELD_MAP = FIELD_MAP;
}

